<?php

namespace App\Http\Controllers\Auth;

use App\Enums\UserStatus;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Rules\CustomRecaptcha;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\View\View;

class AuthenticatedSessionController extends Controller {
    protected $setting;

    public function __construct() {
        $this->setting = cache()->get('setting');
    }
    /**
     * Display the login view.
     */
    public function create(): View {
        return view('client.profile.auth.login');
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(Request $request) {
        $setting = Cache::get('setting');

        $rules = [
            'email'                => 'required|email',
            'password'             => 'required',
            'g-recaptcha-response' => $setting->recaptcha_status == 'active' ? ['required', new CustomRecaptcha()] : '',
        ];

        $customMessages = [
            'email.required'                => __('Email is required'),
            'password.required'             => __('Password is required'),
            'g-recaptcha-response.required' => __('Please complete the recaptcha to submit the form'),
        ];
        $this->validate($request, $rules, $customMessages);

        $credential = [
            'email'    => $request->email,
            'password' => $request->password,
        ];

        $user = User::where('email', $request->email)->first();

        if ($user) {
            if ($user->status == UserStatus::ACTIVE->value) {
                if ($user->is_banned == UserStatus::UNBANNED->value) {
                    if ($user->email_verified_at == null) {
                        $notification = __('Please verify your email');
                        $notification = ['message' => $notification, 'alert-type' => 'error'];

                        return redirect()->back()->with($notification);
                    }

                    if (Hash::check($request->password, $user->password)) {
                        if (Auth::guard('web')->attempt($credential, $request->remember)) {

                            $notification = __('Logged in successfully.');
                            $notification = ['message' => $notification, 'alert-type' => 'success'];

                            $intendedUrl = session()->get('url.intended');
                            if ($intendedUrl && (!Str::contains($intendedUrl, '/admin') && !Str::contains($intendedUrl, '/lawyer'))) {
                                return redirect()->intended(route('dashboard'))->with($notification);
                            }
                            return redirect()->route('dashboard')->with($notification);
                        }
                    } else {
                        $notification = __('Invalid Credentials');
                        $notification = ['message' => $notification, 'alert-type' => 'error'];

                        return redirect()->back()->with($notification);
                    }
                } else {
                    $notification = __('Inactive account');
                    $notification = ['message' => $notification, 'alert-type' => 'error'];

                    return redirect()->back()->with($notification);
                }
            } else {
                $notification = __('Inactive account');
                $notification = ['message' => $notification, 'alert-type' => 'error'];

                return redirect()->back()->with($notification);
            }
        } else {
            $notification = __('Invalid Credentials');
            $notification = ['message' => $notification, 'alert-type' => 'error'];

            return redirect()->back()->with($notification);
        }
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse {
        Auth::guard('web')->logout();

        $notification = __('Logged out successfully.');
        $notification = ['message' => $notification, 'alert-type' => 'success'];

        return redirect()->route('login')->with($notification);
    }
}
